/*
 * ALSA driver for Panasonic UniPhier series.
 * 
 * Copyright (c) 2013 Panasonic corporation.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/spinlock.h>
#include <linux/vmalloc.h>
#include <linux/sched.h>
#include <linux/signal.h>
#include <linux/wait.h>
#include <linux/kthread.h>
#include <linux/delay.h>

#include "mn2ws-pcm.h"

MODULE_AUTHOR("Katsuhiro Suzuki <suzuki.katsuhiro002@jp.panasonic.com>");
MODULE_DESCRIPTION("Panasonic UniPhier PCM Playback Driver");
MODULE_LICENSE("GPL");

static void mn2ws_pcm_play_transfer_sound(struct mn2ws_pcm_dev *d);

static int mn2ws_pcm_play_open(struct snd_pcm_substream *substream);
static int mn2ws_pcm_play_close(struct snd_pcm_substream *substream);
static int mn2ws_pcm_play_ioctl(struct snd_pcm_substream *substream, 
	unsigned int cmd, void *arg);
static int mn2ws_pcm_play_hw_params(struct snd_pcm_substream *substream, 
	struct snd_pcm_hw_params *hw_params);
static int mn2ws_pcm_play_hw_free(struct snd_pcm_substream *substream);
static int mn2ws_pcm_play_prepare(struct snd_pcm_substream *substream);
static int mn2ws_pcm_play_trigger(struct snd_pcm_substream *substream, int cmd);
static snd_pcm_uframes_t mn2ws_pcm_play_pointer(
	struct snd_pcm_substream *substream);
//static int mn2ws_pcm_play_copy(struct snd_pcm_substream *substream, 
//	int channel, snd_pcm_uframes_t pos, void *buf, snd_pcm_uframes_t count);
//static int mn2ws_pcm_play_silence(struct snd_pcm_substream *substream, 
//	int channel, snd_pcm_uframes_t pos, snd_pcm_uframes_t count);
static struct page *mn2ws_pcm_play_page(struct snd_pcm_substream *substream, 
	unsigned long offset);
//static int mn2ws_pcm_play_mmap(struct snd_pcm_substream *substream, 
//	struct vm_area_struct *vma);
//static int mn2ws_pcm_play_ack(struct snd_pcm_substream *substream);

struct snd_pcm_ops mn2ws_pcm_playback_ops = {
	.open      = mn2ws_pcm_play_open, 
	.close     = mn2ws_pcm_play_close, 
	.ioctl     = mn2ws_pcm_play_ioctl, 
	.hw_params = mn2ws_pcm_play_hw_params, 
	.hw_free   = mn2ws_pcm_play_hw_free, 
	.prepare   = mn2ws_pcm_play_prepare, 
	.trigger   = mn2ws_pcm_play_trigger, 
	.pointer   = mn2ws_pcm_play_pointer, 
	//.copy      = mn2ws_pcm_play_copy, 
	//.silence   = mn2ws_pcm_play_silence, 
	.page      = mn2ws_pcm_play_page, 
	//.mmap      = mn2ws_pcm_play_mmap, 
	//.ack       = mn2ws_pcm_play_ack, 
};

int mn2ws_pcm_play_thread(void *data)
{
	struct mn2ws_pcm_dev *d = data;
	unsigned long hw_ptr_s, hw_ptr;
	struct timeval tv_s, tv_e, tv_i;
	int result = 0;
	
	if (d->play.desc->init) {
		//init the DMA
		d->play.desc->init(d);
	}
	
	while (!kthread_should_stop()) {
		//wait for start trigger
		spin_lock(&d->play.spin);
		while (!d->play.start_trigger) {
			spin_unlock(&d->play.spin);
			
			if (kthread_should_stop()) {
				PRINTF_NOTE("ch%d_p: abort kthread\n", d->ch);
				result = 1;
				goto err_out1;
			}
			
			result = wait_event_interruptible(d->play.q, 
				(d->play.start_trigger
					|| kthread_should_stop()));
			if (result != 0) {
				PRINTF_WARN("ch%d_p: wait_event failed.\n", 
					d->ch);
				result = 1;
				goto err_out1;
			}
			
			spin_lock(&d->play.spin);
		}
		d->play.start_trigger = 0;
		
		spin_unlock(&d->play.spin);
		
		
		//setup the DMA
		d->play.desc->setup(d);
		
		//Fill HW buffer with the silence data
		d->play.desc->silence(d, d->play.hw.len - 256);
		
		mn2ws_pcm_dbg_pcmif_hwbuf(d, &d->play);
		mn2ws_pcm_dbg_pcmif_ringbuf(d, &d->play);
		
		//kick the DMA
		d->play.desc->start(d);
		
		
		spin_lock(&d->play.spin);
		//update HW write pointer
		if (d->play.desc->set_devptr) {
			d->play.desc->set_devptr(d, 
				get_wp_ringbuf(&d->play.hw));
		}
		spin_unlock(&d->play.spin);
		
		//wait for the HW read pointer
		do_gettimeofday(&tv_s);
		hw_ptr_s = hw_ptr = d->play.desc->get_hwptr(d);
		while (hw_ptr == 0) {
			hw_ptr = d->play.desc->get_hwptr(d);
		}
		do_gettimeofday(&tv_e);
		timersub(&tv_e, &tv_s, &tv_i);
		DPRINTF("ch%d_p: wait for HW %d.%06d[s]\n", 
			d->ch, (int)tv_i.tv_sec, (int)tv_i.tv_usec);
		
		//update HW buffer read pointer
		spin_lock(&d->play.spin);
		set_rp_ringbuf(&d->play.hw, d->play.desc->get_hwptr(d));
		spin_unlock(&d->play.spin);
		
		mn2ws_pcm_dbg_pcmif_ringbuf(d, &d->play);
		
		//transfer sound data
		mn2ws_pcm_play_transfer_sound(d);
		
		//stop the DMA
		d->play.desc->stop(d);
		
		spin_lock(&d->play.spin);
		d->play.stop_trigger = 0;
		spin_unlock(&d->play.spin);
	}
	
err_out1:
	if (d->play.desc->term) {
		//term the DMA
		d->play.desc->term(d);
	}
	
	return result;
}

static void mn2ws_pcm_play_transfer_sound(struct mn2ws_pcm_dev *d)
{
	struct mn2ws_pcm_chip *chip;
	struct snd_pcm_substream *substream;
	struct snd_pcm_runtime *runtime;
	size_t remain;
	loff_t wp;
	size_t hw_space, alsa_remain, len, tran, writen;
	
	spin_lock(&d->play.spin);
	
	//get ALSA PCM runtime
	chip = d->chip;
	substream = chip->substream;
	runtime = substream->runtime;
	
	spin_unlock(&d->play.spin);
	
	while (!kthread_should_stop()) {
		spin_lock(&d->play.spin);
		
		if (d->play.stop_trigger) {
			spin_unlock(&d->play.spin);
			break;
		}
		
		//get read pointer from HW
		set_rp_ringbuf(&d->play.hw, d->play.desc->get_hwptr(d));
		
		//get write pointer of bounce buffer
		runtime = substream->runtime;
		if (runtime) {
			remain = d->play.alsa.len - frames_to_bytes(runtime, 
				snd_pcm_playback_avail(runtime));
		} else {
			remain = 0;
		}
		remain = min(remain, d->play.alsa.len - 1);
		
		wp = get_rp_ringbuf(&d->play.alsa) + remain;
		set_wp_ringbuf(&d->play.alsa, wp);
		
		//get space of HW buffer
		hw_space = get_space_ringbuf(&d->play.hw);
		hw_space -= hw_space % 256;
		
		//get remain of bounce buffer
		alsa_remain = get_remain_ringbuf(&d->play.alsa);
		alsa_remain -= alsa_remain % 256;
		
		//transfer bounce buffer -> HW buffer
		len = min(hw_space, alsa_remain);
		while (len > 0 && !kthread_should_stop()) {
			tran = len;
			tran -= len % 256;
			
			writen = d->play.desc->copy(d, &d->play.alsa, tran);
			if (writen == 0) {
				break;
			} else if (writen < 0) {
				PRINTF_WARN("ch%d_p: cannot copy to "
					"HW buffer.\n", d->ch);
				break;
			}
			len -= writen;
		}
		
		spin_unlock(&d->play.spin);
		
		//֤߰򹹿
		runtime = substream->runtime;
		if (runtime) {
			snd_pcm_period_elapsed(substream);
		}
		
		//wait event of HW
		d->play.desc->wait_hwevent(d);
		
		mn2ws_pcm_dbg_pcmif_ringbuf_maxmin(d, &d->play);
	}
}

static int mn2ws_pcm_play_open(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct mn2ws_pcm_chip *chip = snd_pcm_substream_chip(substream);
	struct mn2ws_pcm_dev *d = chip->device;
	
	DPRINTF("%s\n", __func__);
	
	spin_lock(&d->play.spin);
	
	//link chip -> substream
	chip->substream = substream;
	
	//determine the snd_pcm_hardware 
	memset(&runtime->hw, 0, sizeof(runtime->hw));
	d->play.desc->hardware(d, &runtime->hw);
	
	spin_unlock(&d->play.spin);
	
	return 0;
}

static int mn2ws_pcm_play_close(struct snd_pcm_substream *substream)
{
	DPRINTF("%s\n", __func__);
	
	return 0;
}

static int mn2ws_pcm_play_ioctl(struct snd_pcm_substream *substream, 
	unsigned int cmd, void *arg)
{
	int err = -ENOTTY;
	int result;
	
	//DPRINTF("%s %d\n", __func__, cmd);
	
	switch (cmd) {
	default:
		//default
		result = snd_pcm_lib_ioctl(substream, cmd, arg);
		break;
	}
	
	return result;
	
//err_out1:
	return err;
}

static int mn2ws_pcm_play_hw_params(struct snd_pcm_substream *substream, 
	struct snd_pcm_hw_params *hw_params)
{
	struct mn2ws_pcm_chip *chip = snd_pcm_substream_chip(substream);
	struct mn2ws_pcm_dev *d = chip->device;
	size_t size;
	int err;
	
	DPRINTF("%s\n", __func__);
	
	size = params_buffer_bytes(hw_params);
	
	err = mn2ws_pcm_pcmif_alloc_bounce_buffer(substream, size, &d->play);
	if (err != 0) {
		PRINTF_WARN("ch%d_p: failed to allocate bounce buffer.\n", 
			d->ch);
		goto err_out1;
	}
	
	return 0;
	
err_out1:
	return err;
}

static int mn2ws_pcm_play_hw_free(struct snd_pcm_substream *substream)
{
	struct mn2ws_pcm_chip *chip = snd_pcm_substream_chip(substream);
	struct mn2ws_pcm_dev *d = chip->device;
	
	DPRINTF("%s\n", __func__);
	
	mn2ws_pcm_pcmif_free_bounce_buffer(substream, &d->play);
	
	return 0;
}

static int mn2ws_pcm_play_prepare(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct mn2ws_pcm_chip *chip = snd_pcm_substream_chip(substream);
	struct mn2ws_pcm_dev *d = chip->device;
	int err;
	
	DPRINTF("%s\n", __func__);
	
	runtime->delay = 0;
	
	mn2ws_pcm_dbg_pcmif_runtime(substream, &d->play);
	
	//Reset HW ringbuf
	err = mn2ws_pcm_pcmif_reset_hw_ringbuf(substream, &d->play);
	if (err != 0) {
		PRINTF_WARN("ch%d_p: failed to reset HW ring buffer.\n", d->ch);
		goto err_out1;
	}
	
	//Reset bounce ringbuf
	err = mn2ws_pcm_pcmif_reset_bounce_ringbuf(substream, &d->play);
	if (err != 0) {
		PRINTF_WARN("ch%d_p: failed to reset bounce ring buffer.\n", 
			d->ch);
		goto err_out1;
	}
	
	return 0;
	
err_out1:
	return err;
}

/**
 * ꡼׶ػ
 */
static int mn2ws_pcm_play_trigger(struct snd_pcm_substream *substream, int cmd)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct mn2ws_pcm_chip *chip = snd_pcm_substream_chip(substream);
	struct mn2ws_pcm_dev *d = chip->device;
	snd_pcm_uframes_t avail;
	int result = -EINVAL;
	
	DPRINTF("%s\n", __func__);
	
	avail = snd_pcm_playback_avail(runtime);
	DPRINTF("ch%d_p: snd_pcm_playback_avail: %d\n", d->ch, (int)avail);
	
	spin_lock(&d->play.spin);
	
	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
		DPRINTF("ch%d_p: start trigger\n", d->ch);
		
		d->play.start_trigger = 1;
		result = 0;
		
		break;
	case SNDRV_PCM_TRIGGER_STOP:
		DPRINTF("ch%d_p: stop trigger\n", d->ch);
		
		d->play.stop_trigger = 1;
		result = 0;
		
		break;
	default:
		//failed
		result = -EINVAL;
		break;
	}
	
	spin_unlock(&d->play.spin);
	
	wake_up_interruptible(&d->play.q);
	
	return result;
}

static snd_pcm_uframes_t mn2ws_pcm_play_pointer(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct mn2ws_pcm_chip *chip = snd_pcm_substream_chip(substream);
	struct mn2ws_pcm_dev *d = chip->device;
	snd_pcm_uframes_t rd_frame;
	
	//DPRINTF("%s: avail:%d\n", __func__, 
	//	(int)snd_pcm_playback_avail(runtime));
	
	spin_lock(&d->play.spin);
	
	//set write pointer of HW buffer
	if (d->play.desc->set_devptr) {
		d->play.desc->set_devptr(d, get_wp_ringbuf(&d->play.hw));
	}
	
	//get read pointer of bounce buffer
	rd_frame = bytes_to_frames(runtime, get_rp_ringbuf(&d->play.alsa));
	
	spin_unlock(&d->play.spin);
	
	//DPRINTF("pointer:%d\n", (int)rd_frame);
	
	return rd_frame;
}

static struct page *mn2ws_pcm_play_page(struct snd_pcm_substream *substream, unsigned long offset)
{
	void *pageptr = substream->runtime->dma_area + offset;

	DPRINTF("%s\n", __func__);

	return vmalloc_to_page(pageptr);
}
